//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.4

import wesual.data.Core 1.0
import wesual.Stage    1.0
import wesual.Create   1.0
import wesual.Ui        1.0
import wesual.Editor    1.0

import wesual.Ui.Private 1.0

import wesual.Controls 1.0 as DeprecatedControls


UiPopupAnchor {
    id : flyout

    signal productsPicked(var products)

    anchors.centerIn : parent
    popup            : popupDelegate

    Component {
        id : popupDelegate

        Rectangle {
            id : menu

            readonly property int bottomDistance :
                  flyout.distanceBottom + (flyout.parent.height / 2)
            readonly property int topDistance :
                  flyout.distanceTop + (flyout.parent.height / 2)
            readonly property bool flipped : topDistance > bottomDistance
            readonly property bool shift   : flyout.distanceRight < width

            function reset() {
                // reset filter
                textInput.text = "";
                productNameFilter.propertyFilter = "";
                textInput.forceActiveFocus();

                // stop the timer because it will be restarted when text is changing
                if (timer.running) {
                    timer.stop();
                }

                // reset list view
                productView.currentIndex      = -1;
                productView.multipleSelection = false;
                productView.selections        = [];
                productView.contentY          = 0;
            }

            PersistentTypeListing {
                id : productListing

                type    : "product:foodservice"
                context : Editor.context
            }

            DocumentListSorter {
                id : productSortedList

                documents : filteredProductListing.resultingDocuments
                sorters   : TypeSorter {
                    property : "name"
                }
            }

            DocumentListFilter {
                id : filteredProductListing

                documents : productListing.documents
                filters   : productNameFilter
            }

            StringFilter {
                id : productNameFilter

                property : "name"
            }

            Connections {
                target : flyout
                onPopupVisibleChanged : {
                    if (flyout.popupVisible)
                        menu.reset();
                }
            }

            implicitWidth  : 302
            implicitHeight : 550

            height :
                flipped ?
                Math.min(topDistance, implicitHeight) :
                Math.min(bottomDistance, implicitHeight)

            color   : UiColors.getColor(UiColors.White)
            opacity : 0

            border {
                width : 1
                color : UiColors.getColor(UiColors.ControlGrey20)
            }

            anchors {
                left : parent.left
                leftMargin : {
                    if (shift) {
                        var leftMargin = - flyout.parent.width / 2;
                        var shiftTerm  =
                            Math.max(-width, -(width - flyout.distanceRight));
                        return leftMargin += shiftTerm;
                    } else
                        return -flyout.parent.width / 2;
                }
                top : flipped ? undefined : parent.bottom
                topMargin : -flyout.parent.height / 2
                bottom : flipped ? parent.top : undefined
                bottomMargin : -flyout.parent.height / 2
            }

            Component.onCompleted : textInput.forceActiveFocus()

            states : [
                State {
                    name : "visible"
                    when : flyout.popupVisible

                    PropertyChanges {
                        target  : menu
                        opacity : 1
                    }
                }
            ]

            transitions : Transition {
                NumberAnimation {
                    property : "opacity"
                    duration : 200
                }
            }

            MouseArea {
                id : filterMouseEvents

                z : -1
                anchors.fill    : parent
                hoverEnabled    : enabled
                acceptedButtons : Qt.AllButtons
            }

            Column {
                anchors {
                    fill    : parent
                    margins : 5
                }

                Rectangle {
                    id : filterTextField

                    height : 26
                    border {
                        width : 1
                        color : UiColors.getColor(UiColors.ControlGrey20)
                    }
                    anchors {
                        left    : parent.left
                        right   : parent.right
                    }
                    states : [
                        State {
                            name : "active"
                            when : textInput.activeFocus
                            PropertyChanges {
                                target : filterTextField
                                border.color :
                                    UiColors.getColor(UiColors.SelectionGreen)
                            }
                        },
                        State {
                            name : "hovered"
                            when : filterMouseArea.containsMouse
                            PropertyChanges {
                                target : filterTextField
                                border.color :
                                    UiColors.getColor(UiColors.HoverGreen)
                            }
                        }
                    ]

                    UiTextContextMenu {
                        textEditor : textInput
                        onContextMenuRequested : textInput.forceActiveFocus()
                    }

                    MouseArea {
                        id : filterMouseArea

                        hoverEnabled : enabled
                        anchors.fill : parent
                        cursorShape  : Qt.IBeamCursor
                        onClicked    : textInput.forceActiveFocus()
                    }

                    UiText {
                        id : placeholder

                        text    : qsTrId("98dcb58ba55d8918")
                        visible : textInput.length === 0
                        font    : UiFonts.getFont(UiFonts.Light, 14)
                        color   : UiColors.getColor(UiColors.MediumGrey33)
                        verticalAlignment : Text.AlignBottom
                        anchors {
                            fill    : parent
                            margins : 4
                        }
                    }
                    TextInput {
                        id : textInput

                        font              : placeholder.font
                        verticalAlignment : Qt.AlignBottom
                        selectByMouse     : true
                        selectionColor    : UiColors.getColor(UiColors.SelectionGreen)
                        selectedTextColor : UiColors.getColor(UiColors.White)
                        clip              : true
                        anchors {
                            fill    : parent
                            margins : 4
                        }
                        onTextChanged : timer.restart()
                        onAccepted    : {
                            var index = productView.currentIndex;

                            if (index === -1)
                                return;

                            var product = productView.model[index];

                            if (productView.multipleSelection) {
                                productView.select(product);
                            } else {
                                productsPicked([ product ]);
                            }
                        }

                        // key navigation
                        Keys.onUpPressed : {
                            productView.hoveredIndex =
                                       Math.max(productView.hoveredIndex - 1, 0)
                            productView.currentIndex = productView.hoveredIndex;
                        }
                        Keys.onDownPressed : {
                            productView.hoveredIndex =
                                    Math.min(productView.hoveredIndex + 1,
                                             productView.model.length - 1);
                            productView.currentIndex = productView.hoveredIndex;
                        }
                        Keys.onTabPressed     : {
                            if (productView.multipleSelection)
                                linkAddProducts.focus = true;
                        }
                        Keys.onBacktabPressed : {
                            if (productView.multipleSelection)
                                linkAddProducts.focus = true;
                        }
                    }

                    // start filtering
                    Timer {
                        id : timer

                        interval : 300
                        running  : false
                        repeat   : false

                        onTriggered : {
                            productNameFilter.propertyFilter = textInput.text;
                        }
                    }
                }

                Item { // spacing
                    width  : 1
                    height : 4
                }

                UiCheckButton {
                    id : btnMultiSelect

                    text      : qsTrId("5849c4a2552fe2e7")
                    iconName  : "multi-selection"
                    bindable  : true
                    checked   : productView.multipleSelection
                    onClicked : {
                        productView.multipleSelection =
                                                 !productView.multipleSelection;

                        if(!productView.multipleSelection)
                            productView.selections = [];
                    }

                }

                Item { // spacing
                    width  : 1
                    height : 4
                }

                Rectangle {
                    height : 1
                    width  : parent.width
                    color  : UiColors.getColor(UiColors.MediumLightGrey27)
                }

                Item { // spacing
                    width  : 1
                    height : 3
                }

                DeprecatedControls.ScrollView {
                    anchors {
                        left  : parent.left
                        right : parent.right
                        leftMargin  : -1
                        rightMargin : -3
                    }
                    height : {
                        var h = menu.height;

                        // filter and button area, margins inclusive
                        h -= (filterTextField.height + btnMultiSelect.height +
                              (4 * 4) + 1);

                        // add area in bottom, margins inclusive
                        h -= linkArea.height

                        if (linkArea.height > 0)
                            h -= 3;

                        // bottom margin
                        h -= 4;

                        return h;
                    }

                    ListView {
                        id : productView

                        property var  selections        : []
                        property bool multipleSelection : false
                        property int  hoveredIndex      : -1

                        function select(product) {
                            var currSelection = selections;
                            var toSelect      = currSelection.indexOf(product);

                            if (toSelect === -1)
                                currSelection.push(product);
                            else
                                currSelection.splice(toSelect, 1);

                            selections = currSelection;
                        }

                        model          : productSortedList.resultingDocuments
                        onModelChanged : currentIndex = -1
                        delegate       : listDelegate
                        highlightFollowsCurrentItem : true
                    }
                }

                Item { // spacing
                    width  : 1
                    height : 3
                }

                Item {
                    id : linkArea

                    width   : parent.width
                    height  : productView.multipleSelection ? 32 : 0
                    visible : height > 0
                    opacity : 32 / height

                    Behavior on height {
                        NumberAnimation { duration : 120 }
                    }

                    Rectangle {
                        height  : 1
                        width   : parent.width
                        color   : UiColors.getColor(UiColors.MediumLightGrey27)
                        opacity : parent.opacity
                    }

                    UiCommandLink {
                        id : linkAddProducts

                        anchors.centerIn : parent
                        anchors.verticalCenterOffset : 5
                        text : qsTrId("14ad8d993707fef6")
                               .arg(productView.selections.length)
                        enabled   : productView.selections.length > 0
                        visible   : parent.visible
                        opacity   : parent.opacity
                        onClicked : productsPicked(productView.selections)

                        Keys.onTabPressed     : textInput.focus = true
                        Keys.onBacktabPressed : textInput.focus = true
                    }
                }
            }
            Component {
                id : listDelegate

                Item {
                    property var view : ListView.view

                    readonly property bool isSelected :
                                       view.selections.indexOf(modelData) !== -1

                    width  : parent ? parent.width : 100
                    height : 58

                    Rectangle {
                        id : selectionRect

                        color        : UiColors.getColor(UiColors.White)
                        anchors.fill : parent

                        UiCheckBox {
                            id : checkBox

                            width    : view.multipleSelection ? implicitWidth : 0
                            text     : ""
                            visible  : width > 0
                            opacity  : width / implicitWidth
                            checked  : isSelected
                            bindable : true
                            anchors {
                                left           : parent.left
                                leftMargin     : 6
                                verticalCenter : parent.verticalCenter
                            }

                            Behavior on opacity {
                                NumberAnimation { duration : 5 }
                            }

                            Behavior on width {
                                NumberAnimation { duration : 70 }
                            }
                        }

                        AssetThumbnail {
                            id : image

                            width  : 50
                            height : width
                            asset  : modelData ? modelData.primaryImage : null
                            anchors {
                                left : checkBox.right
                                leftMargin : !view.multipleSelection ? 0 : 9
                                verticalCenter : parent.verticalCenter
                            }
                        }

                        UiText {
                            id : nameLabel

                            text     : modelData.name
                            wrapMode : Text.Wrap
                            maximumLineCount : 2
                            anchors {
                                left  : image.right
                                right : parent.right
                                leftMargin     : 9
                                verticalCenter : parent.verticalCenter
                            }
                        }

                        MouseArea {
                            id : selectionMouseArea

                            anchors.fill : parent
                            hoverEnabled : enabled

                            onContainsMouseChanged : {
                                if (containsMouse) {
                                    view.hoveredIndex = index;
                                } else {
                                    view.hoveredIndex = -1;
                                }
                            }
                            onPressed : {
                                // if control modifier is pressed multi-selection is activated
                                if (!view.multipleSelection &&
                                    mouse.modifiers === Qt.ControlModifier) {
                                    view.multipleSelection = true;
                                }
                            }
                            onClicked : {
                                view.select(modelData);
                                if (!view.multipleSelection) {
                                    productsPicked([ modelData ]);
                                }
                            }
                        }

                        states : [
                            State {
                                name : "selected"
                                when : isSelected || selectionMouseArea.pressed
                                PropertyChanges {
                                    target : selectionRect
                                    color  : UiColors.getColor(UiColors.SelectionGreen)
                                }
                                PropertyChanges {
                                    target    : nameLabel
                                    highlight : true
                                }
                            },
                            State {
                                name : "hovered"
                                when : view.hoveredIndex === index
                                PropertyChanges {
                                    target : selectionRect
                                    color  : UiColors.getColor(UiColors.HoverGreen)
                                }
                            }
                        ]
                    }
                }
            }
        }
    }
}

